package furny.ga.logger;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import furny.ga.logger.entities.EventType;
import furny.ga.logger.entities.GeneEntry;
import furny.ga.logger.entities.IndividualEntry;
import furny.ga.tuples.Couple;

/**
 * Caches data for the db logger and statistics tool.
 * 
 * @since 11.08.2012
 * @author Stephan Dreyer
 */
public final class DataCache {
  // the single instance of DataCache
  private static final DataCache INSTANCE = new DataCache();

  private enum DataId {
    MEAN_FITNESS_OVER_TIME, MEAN_FITNESS_OVER_GENERATIONS, MIN_FITNESS_OVER_TIME, MAX_FITNESS_OVER_TIME, MIN_FITNESS_OVER_GENERATIONS, MAX_FITNESS_OVER_GENERATIONS, FURNID_COUNT
  }

  private final Map<Couple<?, ?>, Map<Double, Double>> doubleDoubleCache = new HashMap<Couple<?, ?>, Map<Double, Double>>();
  private final Map<Couple<?, ?>, Map<Double, Long>> doubleLongCache = new HashMap<Couple<?, ?>, Map<Double, Long>>();
  private final Map<Couple<?, ?>, Map<Long, Long>> longLongCache = new HashMap<Couple<?, ?>, Map<Long, Long>>();
  private final Map<Couple<?, ?>, Map<Long, Double>> longDoubleCache = new HashMap<Couple<?, ?>, Map<Long, Double>>();

  private DataCache() {
  }

  public static DataCache getInstance() {
    return INSTANCE;
  }

  public Map<Double, Double> getMeanFitnessTime(final Long runId) {
    final Couple<Long, DataId> key = new Couple<Long, DataId>(runId,
        DataId.MEAN_FITNESS_OVER_TIME);

    Map<Double, Double> result = doubleDoubleCache.get(key);
    if (result == null) {
      result = new HashMap<Double, Double>();

      final List<IndividualEntry> list = StatisticsCache.getInstance()
          .getIndiduals(runId, EventType.POPULATION_INITIATED,
              EventType.INDIVIDUALS_INSERTED);

      final Map<Double, Integer> generationIndCount = new HashMap<Double, Integer>();
      final Map<Double, Double> generationFitness = new HashMap<Double, Double>();

      for (final IndividualEntry entry : list) {
        final Double time = (entry.getEvent().getDate().getTime() - entry
            .getEvent().getEvaluationRun().getStarted().getTime()) / 1000d;

        Integer count = generationIndCount.get(time);
        if (count != null) {
          count++;
        } else {
          count = 1;
        }
        generationIndCount.put(time, count);

        Double fitness = generationFitness.get(time);

        final Double newFitness = entry.getFitness() != null ? entry
            .getFitness() : null;
        if (newFitness != null) {
          if (fitness != null) {
            fitness += newFitness;
          } else {
            fitness = newFitness;
          }

          generationFitness.put(time, newFitness);
        }
      }

      for (final Entry<Double, Integer> entry : generationIndCount.entrySet()) {
        Double fitness = null;
        if (generationFitness.get(entry.getKey()) != null) {
          fitness = generationFitness.get(entry.getKey()) / entry.getValue();
        }

        result.put(entry.getKey(), fitness);
      }

      doubleDoubleCache.put(key, result);
    }
    return result;
  }

  public Map<Long, Double> getMeanFitnessGenerations(final Long runId) {
    final Couple<Long, DataId> key = new Couple<Long, DataId>(runId,
        DataId.MEAN_FITNESS_OVER_GENERATIONS);

    Map<Long, Double> result = longDoubleCache.get(key);
    if (result == null) {
      result = new HashMap<Long, Double>();

      final List<IndividualEntry> list = StatisticsCache.getInstance()
          .getIndiduals(runId, EventType.POPULATION_INITIATED,
              EventType.INDIVIDUALS_INSERTED);

      final Map<Long, Integer> generationIndCount = new HashMap<Long, Integer>();
      final Map<Long, Double> generationFitness = new HashMap<Long, Double>();

      for (final IndividualEntry entry : list) {
        final Long generation = Long.valueOf(entry.getGeneration());

        Integer count = generationIndCount.get(generation);
        if (count != null) {
          count++;
        } else {
          count = 1;
        }
        generationIndCount.put(generation, count);

        Double fitness = generationFitness.get(generation);

        final Double newFitness = entry.getFitness();
        if (newFitness != null) {
          if (fitness != null) {
            fitness += newFitness;
          } else {
            fitness = newFitness;
          }

          generationFitness.put(generation, newFitness);
        }
      }

      for (final Entry<Long, Integer> entry : generationIndCount.entrySet()) {
        final Double fitness;
        if (generationFitness.get(entry.getKey()) != null) {
          fitness = generationFitness.get(entry.getKey()).doubleValue()
              / entry.getValue().doubleValue();
        } else {
          fitness = null;
        }

        result.put(entry.getKey(), fitness);
      }

      longDoubleCache.put(key, result);
    }
    return result;
  }

  public Map<Double, Double> getMaxFitnessTime(final Long runId) {
    final Couple<Long, DataId> key = new Couple<Long, DataId>(runId,
        DataId.MAX_FITNESS_OVER_TIME);

    Map<Double, Double> result = doubleDoubleCache.get(key);
    if (result == null) {
      result = new HashMap<Double, Double>();

      final List<IndividualEntry> list = StatisticsCache.getInstance()
          .getIndiduals(runId, EventType.POPULATION_INITIATED,
              EventType.INDIVIDUALS_INSERTED);

      for (final IndividualEntry entry : list) {
        final Double time = (entry.getEvent().getDate().getTime() - entry
            .getEvent().getEvaluationRun().getStarted().getTime()) / 1000d;

        Double fitness = result.get(time);

        final Double newFitness = entry.getFitness() != null ? entry
            .getFitness() : null;

        if (newFitness != null) {
          if (fitness != null) {
            if (newFitness.doubleValue() > fitness.doubleValue()) {
              fitness = newFitness;
            }
          } else {
            fitness = newFitness;
          }

          result.put(time, fitness);
        }

      }

      doubleDoubleCache.put(key, result);
    }
    return result;
  }

  public Map<Long, Double> getMaxFitnessOverGenerations(final Long runId) {
    final Couple<Long, DataId> key = new Couple<Long, DataId>(runId,
        DataId.MAX_FITNESS_OVER_GENERATIONS);

    Map<Long, Double> result = longDoubleCache.get(key);
    if (result == null) {
      result = new HashMap<Long, Double>();

      final List<IndividualEntry> list = StatisticsCache.getInstance()
          .getIndiduals(runId, EventType.POPULATION_INITIATED,
              EventType.INDIVIDUALS_INSERTED);

      for (final IndividualEntry entry : list) {
        final Long generation = Long.valueOf(entry.getGeneration());

        Double fitness = result.get(generation);

        final Double newFitness = entry.getFitness() != null ? entry
            .getFitness() : null;

        if (newFitness != null) {
          if (fitness != null) {
            if (newFitness.doubleValue() > fitness.doubleValue()) {
              fitness = newFitness;
            }
          } else {
            fitness = newFitness;
          }

          result.put(generation, fitness);
        }

      }

      longDoubleCache.put(key, result);
    }
    return result;
  }

  public Map<Long, Double> getMinFitnessOverGenerations(final Long runId) {
    final Couple<Long, DataId> key = new Couple<Long, DataId>(runId,
        DataId.MIN_FITNESS_OVER_GENERATIONS);

    Map<Long, Double> result = longDoubleCache.get(key);
    if (result == null) {
      result = new HashMap<Long, Double>();

      final List<IndividualEntry> list = StatisticsCache.getInstance()
          .getIndiduals(runId, EventType.POPULATION_INITIATED,
              EventType.INDIVIDUALS_INSERTED);

      for (final IndividualEntry entry : list) {
        final Long generation = Long.valueOf(entry.getGeneration());

        Double fitness = result.get(generation);

        final Double newFitness = entry.getFitness() != null ? entry
            .getFitness() : null;

        if (newFitness != null) {
          if (fitness != null) {
            if (newFitness.doubleValue() < fitness.doubleValue()) {
              fitness = newFitness;
            }
          } else {
            fitness = newFitness;
          }

          result.put(generation, fitness);
        }

      }

      longDoubleCache.put(key, result);
    }
    return result;
  }

  public Map<Long, Long> getFurnitureCount(final Long runId) {
    final Couple<Long, DataId> key = new Couple<Long, DataId>(runId,
        DataId.FURNID_COUNT);

    Map<Long, Long> result = longLongCache.get(key);
    if (result == null) {
      result = new HashMap<Long, Long>();

      final List<IndividualEntry> list = StatisticsCache.getInstance()
          .getIndiduals(runId, EventType.POPULATION_INITIATED,
              EventType.INDIVIDUALS_INSERTED);

      for (final IndividualEntry entry : list) {
        for (final GeneEntry gene : entry.getGenes()) {
          final Long furnId = gene.getFurnId();

          Long count = result.get(furnId);
          if (count != null) {
            count++;
          } else {
            count = 1L;
          }
          result.put(furnId, count);

        }
      }

      longLongCache.put(key, result);
    }

    return result;
  }
}
